function [volhist,imgvol] = threedvert2imgvol(varargin)
%This function takes a 3Dvert csv file generated in synapse analysis and
%generates a tiff stack that can be visualized in 3D in imageJ
%Synatax:   [imgvol,volhist] = threedvert2imgvol('dir1','c:\...','files','syn.csv','mode',1)
%Input:     'path' = the directory where the files are
%           'files' = the position filenames in a cell array.
%           'volrad' = In nanometers radius of the volume the size of the 
%               volume will be padded on all sides by 5 slices. Default = 1000.
%           'bin' = by default we are expecting each pixel to be 1nm, so
%               for default synapse size of 1um diameter the number of bins
%               is 1000.
%Output:    imgvol = the matrix of image volume
%           volhist = 3d histogram of data

[pathname,filename,volsz,cfactor] = parse(varargin);  %parse the input

vertices = single(dlmread([pathname,filename{1}],',',1,0));
dataname = filename{1}(1:end-4);

%initiate image volume
volhist = zeros(volsz,'uint8');
centerpt = volsz./2; %we are expecting the volume to be even all across
%go through and map each verticies into the image volume
for i = 1:size(vertices,1)
    curr_vert = round(vertices(i,:).*(100*cfactor));  %grab current vertex convert to scale witht the bins
    curr_pxpos = curr_vert+centerpt;  %negative will move closer to 0
    volhist(curr_pxpos(1),curr_pxpos(2),curr_pxpos(3)) = volhist(curr_pxpos(1),curr_pxpos(2),curr_pxpos(3))+1;  %draw pixel
end

%normalize and stretch dynamic range
imgvol = imnorm(volhist,[],[],1);

mkdir(pathname,dataname);
stk2tiff(imgvol,dataname,[pathname,dataname,filesep]);

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [pathname,files,volsz,cfactor] = parse(input)

pathname = [];
files = [];
volrad = 1000;  %default 1um radius sphere
bins = 1000;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'path'
                    pathname = input{1,i+1};
                case 'files'
                    files = input{1,i+1};
                case 'volrad'
                    volrad = input{1,i+1};
                case 'bin'
                    bins = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end
%calculate volumn size
cfactor = bins/volrad;  %calculate the compression factor in terms of bins in volume
volsz = [bins*2+10, bins*2+10, bins*2+10];  %the volumn size is 

if isempty(pathname)||isempty(files)   %get files and directory
    [files,pathname,filterindex] = uigetfile2({'*.csv','Text files (*.csv)';'*.*','All Files';},...
        'Open position lists','Multiselect','off');
    if ~iscell(files)   %make sure the files array is a cell array
        files = {files};
    end
end